<?php
/**
 * Created by vm.pl
 * User: Rafał Ignaszewski (rafal@vm.pl)
 * Date: 15.01.14 13:58
 */

namespace VM\FinancialStatementsBundle\Services\Reports\Svg;


class SvgChart
{
    /**
     * @var array
     */
    protected $data = array();

    /**
     * @var \DOMDocument
     */
    protected $Svg;

    /**
     * @var \DOMElement
     */
    protected $Root;


    protected $width = 800;

    protected $height = 300;

    protected $scale = 1;

    protected $circleSpace = 100;

    protected $marginTop = 10;
    protected $marginBottom = 50;
    protected $marginLeft = 160;
    protected $marginRight = 60;

    protected $horizontalLineY = 20;

    protected $max = 0;
    protected $min = 0;


    /**
     * @param $data
     * @return string
     */
    public function create($data)
    {
        $this->data = $data;


        $this->init();
        $this->calculateData();

        $this->drawAxes();
        $this->drawPath();
        $this->drawCircles();

        $this->drawLabels();
        $this->drawPriceLabels();



        return str_replace('<?xml version="1.0"?>', '', $this->Svg->saveXML());
    }

    protected function init()
    {
        $this->Svg = new \DOMDocument();
        $this->Svg->formatOutput = true;
        $this->Root = $this->Svg->createElement('svg');
        $this->Svg->appendChild($this->Root);

        $this->Root->setAttribute('height', $this->height);
        $this->Root->setAttribute('width', $this->width);
    }


    protected function calculateData()
    {
        $this->max = 0;
        $this->min = 0;
        foreach($this->data as $data)
        {
            if($data['value'] > $this->max)
            {
                $this->max = $data['value'];
            }
            if($data['value'] < $this->min)
            {
                $this->min = $data['value'];
            }
        }

        $this->scale = ($this->max != 0 || $this->min != 0) ? ($this->height - $this->marginTop - $this->marginBottom) / ($this->max - $this->min) : 1;
        $this->circleSpace = ($this->width - $this->marginRight - $this->marginBottom) / count($this->data);
        $this->horizontalLineY = ($this->max * $this->scale + $this->marginTop);
    }


    protected function drawAxes()
    {
        $GElement = $this->Svg->createElement('g');
        $this->Root->appendChild($GElement);

        $lineWidth = 2;

        // horizontal
        $Line = $this->Svg->createElement('line');
        $Line->setAttribute('x1', $this->marginLeft - $lineWidth / 2);
        $Line->setAttribute('x2', $this->width);
        $Line->setAttribute('y1', $this->horizontalLineY);
        $Line->setAttribute('y2', $this->horizontalLineY);
        $Line->setAttribute('stroke-width', $lineWidth);
        $Line->setAttribute('stroke', 'grey');

        $GElement->appendChild($Line);

        // vertical
        $Line = $this->Svg->createElement('line');
        $Line->setAttribute('x1', $this->marginLeft);
        $Line->setAttribute('x2', $this->marginLeft);
        $Line->setAttribute('y1', $this->marginTop - $this->marginTop);
        $Line->setAttribute('y2', $this->height - $this->marginBottom);
        $Line->setAttribute('stroke-width', $lineWidth);
        $Line->setAttribute('stroke', 'grey');

        $GElement->appendChild($Line);

    }


    protected function drawCircles()
    {
        $GElement = $this->Svg->createElement('g');
        $this->Root->appendChild($GElement);
        $i = 0;
        foreach($this->data as $data)
        {
            $cx = ($i * $this->circleSpace) + $this->marginLeft;
            $Circle = $this->Svg->createElement('circle');
            $Circle->setAttribute('cx', $cx);
            $Circle->setAttribute('cy', $this->horizontalLineY - ($data['value'] * $this->scale));
            $Circle->setAttribute('r', 5.5);
            $Circle->setAttribute('stroke', 'grey');
            $Circle->setAttribute('stroke-width', 0);
            $Circle->setAttribute('fill', '#058dc7');

            $GElement->appendChild($Circle);
            $i++;
        }
    }


    protected function drawPath()
    {
        $GElement = $this->Svg->createElement('g');
        $this->Root->appendChild($GElement);

        $Path = $this->Svg->createElement('path');
        $Path->setAttribute('fill', 'none');
        $Path->setAttribute('stroke-width', 4);
        $Path->setAttribute('stroke', '#058dc7');
        $Path->setAttribute('fill-opacity', 1);

        $i = 0;
        $pathArray = array();
        foreach($this->data as $data)
        {
            $letter = ($i == 0) ? 'M' : 'L';
            $cx = ($i * $this->circleSpace) + $this->marginLeft;
            $cy = $this->horizontalLineY - ($data['value'] * $this->scale);

            $pathArray[] = $letter . $cx .','.$cy;

            $i++;
        }


        $Path->setAttribute('d', implode(',', $pathArray));

        $GElement->appendChild($Path);
    }


    protected function drawLabels()
    {
        $GElement = $this->Svg->createElement('g');
        $this->Root->appendChild($GElement);


        $i = 0;
        foreach($this->data as $data)
        {
            $cx = ($i * $this->circleSpace) + $this->marginLeft;
            $Text = $this->Svg->createElement('text', $data['label']);
            $Text->setAttribute('x', $cx);
            $Text->setAttribute('y', $this->height - $this->marginBottom / 2);
            $Text->setAttribute('text-anchor', 'middle');
            $Text->setAttribute('font-family', 'Verdana');

            $GElement->appendChild($Text);

            $i++;
        }
    }

    protected function drawPriceLabels()
    {
        $GElement = $this->Svg->createElement('g');
        $this->Root->appendChild($GElement);
        $i = 0;

        if($this->max != 0)
        {
            $MaxText = $this->Svg->createElement('text', $this->formatPrice($this->max));
            $MaxText->setAttribute('x', $this->marginLeft - 10);
            $MaxText->setAttribute('y', $this->marginTop + 2);
            $MaxText->setAttribute('text-anchor', 'end');
            $MaxText->setAttribute('font-family', 'Verdana');
            $GElement->appendChild($MaxText);
        }

        if($this->min != 0)
        {
            $MinText = $this->Svg->createElement('text', $this->formatPrice($this->min));
            $MinText->setAttribute('x', $this->marginLeft - 10);
            $MinText->setAttribute('y', $this->height - $this->marginBottom + 2);
            $MinText->setAttribute('text-anchor', 'end');
            $MinText->setAttribute('font-family', 'Verdana');
            $GElement->appendChild($MinText);
        }

        $ZeroText = $this->Svg->createElement('text', 0);
        $ZeroText->setAttribute('x', $this->marginLeft - 10);
        $ZeroText->setAttribute('y', $this->horizontalLineY + 6);
        $ZeroText->setAttribute('text-anchor', 'end');
        $ZeroText->setAttribute('font-family', 'Verdana');
        $GElement->appendChild($ZeroText);
    }


    protected function formatPrice($price)
    {
        return number_format($price, 2, ',', ' ');
    }
}